/*******************************************************************
*  MINARET (for edge-triggered circuits)
*
*  BY
*
*  NARESH MAHESHWARI AND SACHIN S. SAPATNEKAR
*
*  Copyright 1998 Iowa State University Research Foundation, Inc.
*  All Rights Reserved
*
*  Source Code for retiming edge-triggered circuits
*
*  ASTRA:   Refer to paper in DAC 95 by Deokar & Sapatnekar
*                       and IEEE Tran on CAD 10/1998
*  Minaret: Refer to paper in DAC 97 by Maheshwari & Sapatnekar
*                       and IEEE Tran on VLSI 10/1998
*
*  Contact Address: sachin@ece.umn.edu
*
*        Availiable on as is basis, with no support
*******************************************************************/
/*******************************
FILE bell.c
Contains functions for finding optimal period by binary search,
Bellman Ford algorithm is used at each step of binary search
******************************/

#include "include.h"


/*******************************
GLOBAL VARS
******************************/

extern int db_opt;
extern int db_bin;
int Latch_Patch[4000];
int Latch_Patch_Index =0;


/****************************************
   Bellman Ford  for finding postive cycles
   & single source longest path
   
   uses Shenoy's/Symanski's  predecssor-cycle method to get cycles;
   returns 0 if no cycle found
   else returns total cycle wt/no of nodes in cycle
   i.e. the cycle clock period, which is used a new lower bound

cur_p is the current period, it is used to reweight the edges;
init_value  is the value to which skews are initialized (0 for minperiod; infinity for minarea)
Graph_Dir decides if forward or backward graph will be used for ASAP/ALAP
******************************************************/

float bellford(float cur_p,float init_value,int Graph_Dir)
{
  int i,v,v1,v2,l;
  EDGE *ep;
  float cycle_wt;
  float cycle_p;
  int cycle_node;
  int cycle_count;
  int *mark;
  int *pred;
  int *cycle;
  float *predwt;
  int relax;
	
#ifdef DEBUG
  if(db_opt)printf("entered bellford with %f init val %f dir %d\n",cur_p,init_value,Graph_Dir);
#endif
 
  /* initilazation */

  cycle_p = 0;
  /**** Need the array to be zero so use Calloc (and not malloc) **/
  mark = (int *)CALLOC(noverts+2,sizeof(int));
  predwt = (float *)CALLOC(noverts+2,sizeof(float));
  cycle = (int *)CALLOC(noverts+2,sizeof(int));
	
  /** initilize to -1 ***/
  pred = (int *)MALLOC(noverts+2,sizeof(int));

  for(i=0;i<noverts;i++){
#ifdef SAFE
    cycle[i] = 0;
    mark[i] = 0;
    predwt[i] = 0;
#endif		
    pred[i] = -1;
    verts[i].skew = init_value;  
    /*** update the edge weight ***/
    for(ep= verts[i].edges[Graph_Dir];ep!=NULL;ep= ep->next)
      ep->weight = ep->delay - cur_p;
  }

  verts[0].skew = 0;

  /*** actual relaxation **/
  for(i=0;i<noverts;i++)
    {
      relax = 0;

      for(v=0;v<noverts;v++)
	{
	  for(ep= verts[v].edges[Graph_Dir];ep!=NULL;ep= ep->next)
	    {
	      v1= ep->tovert;
	      if(verts[v1].skew < (verts[v].skew + ep->weight))
		{ /* relax edge ep */
		  relax = 1;
		  verts[v1].skew  = (verts[v].skew + ep->weight);
		  pred[v1] = v;
		  predwt[v1] = ep->delay;
		}
	    }
	}
      if(!(relax))break; /* if no relax done in this iter then done */
    }

#ifdef DEBUG
  if(0)for(l=0;l<noverts;l++)
    {
      printf("%s ie %d pred %d mrk %d cyc %d wt %g\n",nodes[verts[l].nodeid]->name,l,pred[l],mark[l],cycle[l],predwt[l]);
    }
#endif

  /* check for positive cycles */
  cycle_count = 0;
  for(v=0;v<noverts;v++)
    {
      if(!mark[v]){
	v1 = v;
	while(pred[v1] !=  -1)
	  {
	    if((mark[v1] !=0)&&(mark[v1] != (v+1)))break;
	    if(mark[v1] == ( v + 1))
	      {/* cycle detected */
		cycle_wt = predwt[v1];
		cycle_node = 1;	/* may not hold for level latches */
		v2 = pred[v1];
					
		while( v2 != v1)
		  { /* for the cycle */
#ifdef DEBUG
		    if(db_opt)printf(" %s \t", nodes[verts[v2].nodeid]->name);
#endif
		    cycle_wt += predwt[v2];
		    cycle_node++;
		    v2 = pred[v2];
		  }
#ifdef DEBUG
		if(db_opt)printf(" cycle detected at %s\n",nodes[verts[v1].nodeid]->name);
		if(db_opt)printf(" cycle wt %g nodes %d\n",cycle_wt,cycle_node);
#endif
		cycle_p = cycle_wt/cycle_node;
		break;
	      }
	    mark[v1]= v + 1;
	    v1 = pred[v1];
	  }
      }
      if(cycle_p)break;
    }
	
  free(mark);
  free(predwt);
  free(cycle);
  free(pred);
  return(cycle_p);
}

/****************************************
opt_period
   does the actual binary search
   and retuns the optimal clock period found 
****************************************/

float opt_period()
{
  float lowerbd,upperbd,period,cycle_period;
  int iter =0;
  int period_valid;

  /*** set lower and upper bound for binary search **/
  if(Max_self_loop < Mindel)lowerbd = Mindel;
  else lowerbd = Max_self_loop;
  upperbd = Maxdel;
  period = (upperbd +lowerbd)/2;

#ifdef DEBUG
  if(db_bin)printf("INITIAL  U %g L %g \n",upperbd,lowerbd);
#endif

  period_valid = 0;
  /* do bellford atleast once */
  while((upperbd -lowerbd > EPSILON)||(!period_valid))
    {
      period_valid = 1;
      iter++;
      if(db_bin)printf(" IT %d trying period %g U %g L %g\n",iter,period,upperbd,lowerbd);

      cycle_period = bellford(period,0,FORWARD);

#ifdef DEBUG
      if(db_bin)printf(" Period %g INfeasiable \n",period);
#endif

      if(cycle_period == 0)
	{
	  upperbd = period;
	  period = (upperbd +lowerbd)/2;
	}
      else
	{			
	  if(lowerbd < cycle_period)
	    { 
	      lowerbd = cycle_period;
	      period = cycle_period;
	    }
	  else
	    { 
	      lowerbd = cycle_period;
	      period = (upperbd +lowerbd)/2;
	    }
	}
    }
  counter = iter;  /* pass no of it to main */

  /*******************  IMPORTANT *********
    RUN BELLFORD ONCE MORE TO INSURE THAT THE SKEWS ARE
    THE ONE CORRESPONDING TO THE OPTIMAL PERIOD
    AND NOT WHAT ARE LEFT FROM THE LAST RUN
    **************************************/
  cycle_period = bellford(period,0,FORWARD);
	
  return(period);
}

/*****************************************
normalise_skew
  normalize skews to make primary inputs and outputs zero skew
  transfers skews to latches
  Graph_Dir tells FORWARD/REVERSE (ASAP/ALAP)
  skew_mode tells which of "skew/other_skew" to use
       in the latch (for min area)
*****************************************/
void normalise_skews(int Graph_Dir,int skew_mode)
{
  int i,n;
  float pioskew;
  int sign;
  Q *q1;
  NODETYPE *np;

  /** need to transfer negative of the skew for reverse graph */
  if(Graph_Dir == FORWARD)sign = 1;
  else if(Graph_Dir == REVERSE)sign = -1; 
  else {
    printf(" ERR INVALID GRAPH DIR %d  given\n",Graph_Dir);
    exit(0);
  }

  /* normalize skews */
  pioskew = verts[0].skew;
  if(pioskew!= 0)
    for(i=0;i<noverts;i++)
      {
	verts[i].skew =verts[i].skew- pioskew;
      }
  /******************* 
    transfer skews to latches 
    don't do it for PIO node as no latch corresponds to it 
    *********************8*****/
  PA_max_skew = -INF;
  PA_min_skew = INF;
  if(skew_mode == 0)
    {
      for(i=1;i<noverts;i++)
	{
	  n = verts[i].nodeid;

	  nodes[n]->skew = sign*verts[i].skew;
#ifdef STATS
	  if(verts[i].skew <PA_min_skew)PA_min_skew = verts[i].skew;
	  if(verts[i].skew > PA_max_skew)PA_max_skew = verts[i].skew;
#endif
	}
    }
  else
    {
      for(i=1;i<noverts;i++)
	{
	  n = verts[i].nodeid;

	  nodes[n]->other_skew = sign*verts[i].skew;
#ifdef STATS
	  if(verts[i].skew <PA_min_skew)PA_min_skew = verts[i].skew;
	  if(verts[i].skew > PA_max_skew)PA_max_skew = verts[i].skew;
#endif
	}
    }

#ifdef STATS
  if(PA_max_skew <0)PA_diff_skew = -1*PA_min_skew;
  else if(PA_min_skew >0)PA_diff_skew = PA_max_skew;
  else PA_diff_skew = PA_max_skew - PA_min_skew;
#endif

  /*********** just setting any unreachable latches to 0 skew ****
    this is no prob in min period where init val is 0 but in minarea
    init val is -INF so FFs move for ever.
    the code does not expect to have these skews; hence it used in
    SAFE mode only.
    This can be probably corrected by changing the way in which latches are 
    moved in Phase B
    ***************************************************************/
  /*** checking if there is any unreachable latch **/
for(q1=latches;q1 != NULL;q1 = q1->next)
    {
      n = q1->el;
      np = nodes[n];
      if((np->skew > NEARINF)||(np->skew < -NEARINF)){
	printf("Error: the circuit may contain unreachable latches. \n Current version cannot handle circuits with unreachable latches \n\n");
	exit(-1);
      }
    }

#ifdef SAFE	
  for(q1=latches;q1 != NULL;q1 = q1->next)
    {
      n = q1->el;
      np = nodes[n];
      if((np->skew > 1000)||(np->skew < -1000)){
	printf("WARNING skew %f too high setting to 0 [%d]%s\n",np->skew,n,np->name);
	np->skew =0; 
	Latch_Patch[Latch_Patch_Index++]= n;
      }
    }
#endif

}

/*********EOF**********/
