/*******************************************************************
*  MINARET (for edge-triggered circuits)
*
*  BY
*
*  NARESH MAHESHWARI AND SACHIN S. SAPATNEKAR
*
*  Copyright 1998 Iowa State University Research Foundation, Inc.
*  All Rights Reserved
*
*  Source Code for retiming edge-triggered circuits
*
*  ASTRA:   Refer to paper in DAC 95 by Deokar & Sapatnekar
*                       and IEEE Tran on CAD 10/1998
*  Minaret: Refer to paper in DAC 97 by Maheshwari & Sapatnekar
*                       and IEEE Tran on VLSI 10/1998
*
*  Contact Address: sachin@ece.umn.edu
*
*        Availiable on as is basis, with no support
*******************************************************************/
/*******************************
FILE cnvt.c

converts circuit to C-graph and back

Is a bit tricky, only latch information is stored independently;
the latch free C-graph is updated to restore latch poitions.
******************************/

#include "include.h"


/*******************************
GLOBAL VARS
******************************/
int no_gates_to_do=0;
SAVE_LATCH *stored_latches;



/*******************************
ckt2graph

NEEDS TO HAVE ALL SKEWS in latches before it is run;
for each pair of gate with one or more latches in between 
make a entry in the array SAVE_LATCH *stored_latches;
then copy all skews on latches in ASAP and ALAP 
STILL NEED TO COPY the needed (asap or alap) skews to edges
******************************/

void ckt2graph()
{
	int i,n,n1,prev,done;
	NODETYPE *np, *np1;
	Q  *qlatch;
	W *w1, *w2;
	int *gates_to_do;
	F *f1;

	gates_to_do = (int *)MALLOC( nolatches,maxfanout*sizeof(int));  /* gate can occur twice in list; actully as many as sum of all latch fanouts */
	if(gates_to_do == NULL){printf("ERR OUT OF MEM gates_to_do \n");exit(-1);}

	for(i=0;i<nogates;i++)
	{ 
		n=gates[i];
		np =nodes[n];
		for(w1=np->fout;w1 != NULL; w1= w1->next){
			w1->wt = 0;
			w1->list = NULL;
		}
		for(w1=np->fin;w1 != NULL; w1= w1->next){
			w1->wt = 0;
			w1->list = NULL;
		}
	}


	for(qlatch = latches; qlatch!= NULL; qlatch= qlatch->next)
	{
		for(w1= nodes[qlatch->el]->fout;w1 != NULL;w1= w1->next)if(nodes[w1->el]->type != Latch)gates_to_do[no_gates_to_do++]
		    = w1->el;
	}

#ifdef DEBUG
	if(0){
		printf(" %d Gates to do are ",no_gates_to_do);
		for(i=0;i<no_gates_to_do;i++)printf(" %d ",gates_to_do[i]);
		printf(" \n\n");
	}
#endif

	stored_latches = (SAVE_LATCH *)MALLOC(no_gates_to_do,sizeof(SAVE_LATCH));
	if(stored_latches == NULL){ printf("ERR OUT OF MEM for stored_latches size %d \n",no_gates_to_do*sizeof(SAVE_LATCH));exit(-1);
				  }
	for(i=0;i<no_gates_to_do;i++)
	{
		n = gates_to_do[i];
		np = nodes[n];
		stored_latches[i].to = n;
		stored_latches[i].wt = 0;
		stored_latches[i].skews[0] = NULL;
		stored_latches[i].skews[1] = NULL;
		for(w1=np->fin;w1 != NULL; w1= w1->next)
		{
			n1 = w1->el;
			np1 = nodes[n1];
			if(np1->type == Latch) /* go till u get a gate */
			{
				while(np1->type == Latch)
				{
					prev = n1;
					stored_latches[i].wt++;
					f1 = (F *)MALLOC(1,sizeof(F));
					f1->next = stored_latches[i].skews[0];
					f1->val = np1->skew;
					stored_latches[i].skews[0] = f1;

					f1 = (F *)MALLOC(1,sizeof(F));
					f1->next = stored_latches[i].skews[1];
					f1->val = np1->other_skew;
					stored_latches[i].skews[1] = f1;

					n1 = np1->fin->el;
					np1 = nodes[n1];
				}

				/* update fanin of "n" to point to gate n1 */
				w1->el = n1;
				w1->wt = stored_latches[i].wt;
				stored_latches[i].from = n1;
/**************************
!!!!! now update fanout of "n1" from "prev" to "n"
 ADD FANOUTS if needed ie if any latch had fout >1 
(can find new nofout by summing fanouts of latches on the path)
!!!!! since I update fanout to point to gate, next time I won't find the latch I came through 
***************************/
				done = 0;
				for(w2=np1->fout;w2 != NULL; w2= w2->next)
				{
					if(w2->el == prev)
					{
						w2->el = n; 
						w2->wt = stored_latches[i].wt; 
						w2->list = NULL;
						done =1; 
						break;
					}
				}
				if(done == 0)
				{ /** need to add fanout */
					w2 = (W *)MALLOC(1,sizeof(W));
					w2->next = np1->fout;
					w2->el = n;
					w2->wt = stored_latches[i].wt;
					w2->list = NULL;
					np1->fout = w2;
					np1->nofout++;
				}

				break; /* only one fanin at a time */

			} /* if latch */
		} /* for all fanins */

	} /* all gates to do */
	free(gates_to_do);
	

}

/*********************************************
proc_fin(int n)
*************************************************/
void proc_fin(int n)
{
  int in;
  NODETYPE  *np;
  W *w1;

  np = nodes[n];
  for(w1=np->fin;w1 != NULL; w1= w1->next)
    {
      if(nodes[w1->el]->type == Latch)
	{
	  in = w1->el;
	  while(nodes[in]->type == Latch)
	    {
	      in = (nodes[in]->fin)->el;
	    }
	  w1->el = in;
	}
    }
}


/******************************************************
 remove_pio

this functions removes the Primary inputs and outputs (artifically added 
when parsing the netlist).
********************************************************/
void remove_pio()
{
  NODETYPE *np, *nh;
  int n,i,host_fout;
  W *w1,*w2, *wt, *last;


  /****************************
    host in and out are stored as gates with valid node ids, after all gates
    get a free node id for HOST
    *************************/

  if(nofreenodes >0) /* a freed node availiable */
    {
      n = freenodes[nofreenodes--];
      if(validnode[n] !=0){
	printf("\n ERR{add_latch} overwriting node %d nofreenodes %d and are\n\n",n,nofreenodes);
	for(i=0;i<nofreenodes;i++)printf(" %d ",freenodes[i]);
	exit(0);
      }
    }
  else {
    n =maxid++;
  }

  HOST = n;
  nonodes++;
  gates[nogates++] = HOST;
  validnode[HOST] = 1;
  nodes[HOST] = (NODETYPE *)malloc(1*sizeof(NODETYPE));

  last = NULL;
  host_fout = 0;
  nh = nodes[HOST];
  nh->id = HOST;
  strcpy(nh->name,"HOST");
  nh->type = Host;
  nh->skew = 0;
  nh->phase = 0;
  nh->level = -INF;
  nh->nai = 0;
  nh->mindel = 0;
  nh->maxdel = 0;
  nh->minto = 0;
  nh->minti = 0;

  for(i=0;i<nopin;i++)
    { 
      n =primary_input[i];
      np =nodes[n];
      for(w1=np->fout;w1 != NULL; w1= w1->next)
	{
	  /* correct the fanin of fanouts of PI */
	  for(w2=nodes[w1->el]->fin;w2 != NULL;w2= w2->next)
	    {
	      if(w2->el == n)w2->el = HOST;
	    }

	  wt= (W *)CALLOC(1,sizeof(W));
	  wt->el = w1->el;
	  wt->wt = w1->wt;
	  wt->list = copy_F(w1->list);
	  wt->next = last;
	  last = wt;
	  host_fout++;
	}
    }
  nh->nofout = host_fout;
  nh->fout = last;
  last = NULL;
  host_fout = 0;

  for(i=0;i<nopout;i++)
    { 
      n =primary_output[i];
      np =nodes[n];
      for(w1=np->fin;w1 != NULL; w1= w1->next)
	{
	  /* correct the fanouts  of fanin of PO */
	  for(w2=nodes[w1->el]->fout;w2 != NULL;w2= w2->next)
	    {
	      if(w2->el == n)w2->el = HOST;
	    }

	  wt= (W *)CALLOC(1,sizeof(W));
	  wt->el = w1->el;
	  wt->wt = w1->wt;
	  wt->list = copy_F(w1->list);
	  wt->next = last;
	  last = wt;
	  host_fout++;
	}
    }
  nh->nofin = host_fout;
  nh->fin = last;
}


/************************************
 restore_pio

ASSUMES that PIO have correct fanin  & fanout list 
with correct weights since back connection from gates to pio
are lost (only know that they were connected to PIO
NOT to which)
**************************************/
void restore_pio()
{
  NODETYPE *np;
  int n1,n,i,done;
  W *w1,*w2;


  nogates--;
  /** removing host node from gates without deleteing it ***/

  for(i=0;i<nopin;i++)
    { 
      n =primary_input[i];
      np =nodes[n];
      for(w1=np->fout;w1 != NULL; w1= w1->next)
	{
	  /* correct the fanin 
	     of fanouts of PI */
	  done = 0;
	  for(w2=nodes[w1->el]->fin;w2 != NULL;w2= w2->next)
	    {
	      if(w2->el == HOST){
		w2->el = n;
		w2->wt = w1->wt;
		done =1;
		break;
	      }
	    }
	  /* break important so as not to create extra edge **/
	  if(done != 1){
	    printf(" ERR HOST not found in %d\n",w1->el);
	  }

	}
    }
  for(i=0;i<nopout;i++)
    { 
      n =primary_output[i];
      np =nodes[n];
      for(w1=np->fin;w1 != NULL; w1= w1->next)
	{
	  /* correct the fanouts  of fanin of PO */
	  done = 0;
	  for(w2=nodes[w1->el]->fout;w2 != NULL;w2= w2->next)
	    {
	      if(w2->el == HOST){
		w2->el = n;
		w2->wt = w1->wt;
		done =1;
		break;
	      }
	    }
	  /* break important; so as not to create extra edge **/
	  if(done != 1){
	    printf(" ERR HOST not found in %d\n",w1->el);
	  }
	}
    }

}


/**************************************
graph2ckt

puts the latches back in the C-graph to get the circuit

Prequiste: graph has pio and all old latches have been freed
**************************************/
void graph2ckt()
{
  NODETYPE *np;
  int n,i,j,toadd;
  int in,out,new,added=0;
  W *w1;
  char new_name[100];
  F *f1;


  /********************
    works only on one fanin OR fanout since graph is bidirectional;
    (i.e, fanin point to fanout and fanout points to fanin)
    put latches only on fanout to avoid putting duplicate latches
    *************************/
	
  nolatches =0;/** init it */
  for(i=0;i<nopin;i++)
    { 
      n =primary_input[i];
      np =nodes[n];
      for(w1=np->fout;w1 != NULL; w1= w1->next)
	{
	  toadd = w1->wt;
	  if(toadd <0){
	    printf(" ERR graph2ckt negative wt on edge between [%d]%s and %d[%s] wt %d\n",n,np->name,w1->el,nodes[w1->el]->name,toadd);
	    exit(0);
	  }

	  if(toadd > 0)
	    {/**  add  toadd latches */
	      in = n;
	      out = w1->el;
#ifdef DEBUG
	      if(0)printf(" adding %d latches between  [%d]%s  and [%d]%s skew %f\n",toadd,n,np->name,out,nodes[out]->name,w1->list->val);
#endif
	      f1 = w1->list;
	      for(j=0;j<toadd;j++)
		{
		  sprintf(new_name,"LATCH_%d",added++); /* generate a name */
		  /******************
		    !!!!!! for some reason giving f1->val (rather than copying ti new_name;
		    caused new_name not to be passed check prototyping
		    *******************/
		  new = add_latch(in,out,0,new_name);
		  f1 = f1->next;
		  in = new;

		}
	    }
	}
    }


  /** do same thing on all gates **/
  for(i=0;i<nogates;i++)
    { 
      n =gates[i];
      np =nodes[n];
      for(w1=np->fout;w1 != NULL; w1= w1->next)
	{
	  toadd = w1->wt;
	  if(toadd <0){
	    printf(" ERR negative wt on edge between [%d]%s and %d[%s] wt %d\n",n,np->name,w1->el,nodes[w1->el]->name,toadd);
	    exit(0);
	  }

	  if(toadd > 0)
	    {/**  add  toadd latches */
	      in = n;
	      out = w1->el;
#ifdef DEBUG
	      if(0)printf(" adding %d latches between  [%d]%s  and [%d]%s\n",toadd,n,np->name,out,nodes[out]->name);
#endif

	      for(j=0;j<toadd;j++)
		{
		  sprintf(new_name,"LATCH_%d",added++); /* generate a name */
		  new = add_latch(in,out,0,new_name);
		  in = new;
					
		}
	    }
	}
    }


	
}


/************************************
restore_skews(int index)

restores the location and skews of latches on the C-graph
this information was in "stored_latches"
index: which skew (FORWARD/REVERSE) to use
**************************************/

void restore_skews(int index)
{
  int i,n1,n2;
  NODETYPE *np1,*np2;
  W *w1;

  for(i =0;i<no_gates_to_do;i++)
    {
      n1 = stored_latches[i].from;
      np1 = nodes[n1];
      n2 = stored_latches[i].to;
      np2 = nodes[n2];
      /* need to compare weights also, since 2 gates may have multiple edges (through 1 or more latches)  between them with but with different weights **/
      for(w1= np2->fin;w1!= NULL;w1 = w1->next)
	{
	  if((w1->el ==  n1)&&(w1->wt == stored_latches[i].wt ))
	    {
	      w1->list = copy_reverse_F(stored_latches[i].skews[index]);
	      w1->wt = stored_latches[i].wt;
	      break;
	    }
	}
      for(w1= np1->fout;w1!= NULL;w1 = w1->next)
	{
	  if((w1->el ==  n2)&&(w1->wt == stored_latches[i].wt ))
	    {
	      w1->list = stored_latches[i].skews[index];
	      w1->wt = stored_latches[i].wt;
	      stored_latches[i].skews[index] = NULL;
	      break;
	    }
	}
    }
}

/************************************
restore_ckt_graph(int index)

Updates the weight & skews on edges 
(i.e., the location and skew of latches)
index < 0 NO skews only wts updated
index = 0 skew[0] used, i.e., skew
index = 1 skew [1] used, i.e., other_skew
**************************************/

void restore_ckt_graph(int index)
{
  int i,n1,n2;
  NODETYPE *np1, *np2;
  W *w1;

  clear_ckt_graph();

  /************************
    mark all edges as 0 so that I do not update the same edge twice if there are duplicate edges  ASSUMING saved latches have weights > 0
    *****************************/

  for(i =0;i<no_gates_to_do;i++)
    {
      n1 = stored_latches[i].from;
      np1 = nodes[n1];
      n2 = stored_latches[i].to;
      np2 = nodes[n2];
      for(w1= np1->fout;w1!= NULL;w1 = w1->next)if(w1->el == n2)w1->wt = 0;
      for(w1= np2->fin;w1!= NULL;w1 = w1->next)if(w1->el == n1)w1->wt = 0;
    }

  for(i =0;i<no_gates_to_do;i++)
    {
      n1 = stored_latches[i].from;
      np1 = nodes[n1];
      n2 = stored_latches[i].to;
      np2 = nodes[n2];

      /* update wts on fins of n2 */
      for(w1= np2->fin;w1!= NULL;w1 = w1->next)
	{
	  if((w1->el == n1)&& (w1->wt == 0))
	    {
	      if(index > -1)w1->list = copy_reverse_F(stored_latches[i].skews[index]);
	      w1->wt = stored_latches[i].wt;
	      break;
	    }
	}
      /* update wts on fout of n1 */
      for(w1= np1->fout;w1!= NULL;w1 = w1->next)
	{
	  if((w1->el == n2)&& (w1->wt == 0))
	    {
	      if(index > -1)
		{
		  w1->list = stored_latches[i].skews[index];
		  stored_latches[i].skews[index] = NULL;
		}
	      w1->wt = stored_latches[i].wt;
	      break;
	    }
	}

    }
}

/************************************
clear_ckt_graph

Just free the skew list on every edge
**************************************/

void clear_ckt_graph()
{
  int i,n1;
  NODETYPE *np1;
  W *w1;

  for(i =0;i<nogates;i++)
    {
      n1 = gates[i];
      np1 = nodes[n1];

      for(w1= np1->fout;w1!= NULL;w1 = w1->next)
	{
	  w1->wt = 0;
	  if(w1->list != NULL){
	    free_F(w1->list); 
	    w1->list = NULL;
	  }
	}

      for(w1= np1->fin;w1!= NULL;w1 = w1->next)
	{
	  w1->wt = 0;
	  if(w1->list != NULL){
	    free_F(w1->list); 
	    w1->list = NULL;
	  }
	}


    }
}

/************************************
print_saved_latches

simply print the "stored_latches" array for degbugging
**************************************/
void print_saved_latches()
{
  int i;
  F *f1;

  for(i=0;i<no_gates_to_do;i++)
    {
      printf("index %d From %d To %d Wt %d ",i,stored_latches[i].from,stored_latches[i].to,stored_latches[i].wt);
      printf(" skew[0] ");
      for(f1 = stored_latches[i].skews[0]; f1 != NULL;f1 = f1->next)printf(" %.1f  ",f1->val);
      printf(" skew[1] ");
      for(f1 = stored_latches[i].skews[1]; f1 != NULL;f1 = f1->next)printf(" %.1f  ",f1->val);
      printf(" \n");
    }
}

/************************************
set_up_retime_Q

Will reset nao & nai to 0 and then
increment "nao" for every latch at output
ASSUMES that nofout & nofanin are correct and teh queue is clear
**************************************/
void set_up_retime_Q()
{
  int i,n;
  NODETYPE *np;
  for(i=0;i<nogates;i++)
    {
      np = nodes[gates[i]];
      np->nai = 0;
      np->nao = 0;
    }

  top=bot=NULL;
  for(i=0;i<no_gates_to_do;i++)
    {
      n = stored_latches[i].to;
      np = nodes[n];
      np->nai++;
      if(np->nai == np->nofin)addq(n);

    }
}

/************************************
 free_saved_latches
************************************/
void free_saved_latches()
{
  no_gates_to_do =0;
  free(stored_latches);
}


/************************************
update_skews_with_slack_for_reverse_retiming(int index, float period)

!!!!!!! THIS IS A PATCH TO AVOID MULTIPLYING FFs BEFORE BELLMAN-FORD
 looks at maxti of fanout of all latches & updates skews[index]
**************************************/
void update_skews_with_slack_for_reverse_retiming(int index, float period)
{
  int i,n,j;
  F *f1;
  float old_skew, new_skew;
	
  for(i=0;i<no_gates_to_do;i++)
    {
      n = stored_latches[i].to;
      f1 = stored_latches[i].skews[index];
      for(j=0;j<(stored_latches[i].wt -1);j++)f1 = f1->next;
      old_skew = f1->val;
      new_skew = period - nodes[n]->rev_maxto;
      if(new_skew < old_skew){
	printf("ERR latch at input of [%d]%s has skew %g but Period %g rev_maxto %g \n",n,nodes[n]->name,old_skew,period,nodes[n]->rev_maxto);
      }

      f1->val = new_skew;

    }

}

/************************************
 to be tested, not in use
To directly add shared latches at a gate fanout;
is a better way than adding single output latches and then merging them 
************************************/
int add_merged_latches_at_fanout(int in)
{

  struct newlatch_struct
  {
    int latch_id;
    W *fanout_list;
    int nofanouts;
  } *outlist; /* [i] keeps fanouts with wt i so [0] has one with no latches */

  int n,n1,i,wt;
  NODETYPE *np,*np1,*npin;
  int maxw;
  W *w1,*w2,*w3;
  Q *q1;
  char name[100];

  npin = nodes[in];

  maxw =0;
  for(w1= npin->fout;w1 != NULL; w1 = w1->next)
    {
      if(w1->wt <0){printf(" ERR graph2ckt negative wt on edge between [%d]%s and %d[%s] wt %d\n",n,np->name,w1->el,nodes[w1->el]->name,w1->wt);exit(0);}
      if(w1->wt > maxw) maxw = w1->wt;
    }

  if(maxw == 0)return(0); /* no need to do any thing */
  /*** need one for 0 wt also */
  outlist = (struct newlatch_struct *)MALLOC(maxw+1,sizeof(struct newlatch_struct));

  /*** allocate mem and get ids for latches [1] to (inc) [maxw]***/
  for(i=1;i<=maxw;i++)
    {
      if(nofreenodes >0) /* a freed node availiable */
	{
	  n = freenodes[nofreenodes--];
	  if(validnode[n] !=0)
	    {
	      printf("\n ERR{add_merged_latches_at_fanout} overwriting node %d\n\n",n);
	      exit(-1);
	    }
		
	}
      else 
	{
	  n = maxid++;
	}

      outlist[i].latch_id = n;
      outlist[i].nofanouts = 0;
      outlist[i].fanout_list = NULL;

      np = (NODETYPE *)CALLOC(1,sizeof(NODETYPE));
      nodes[n] = np;
      np->id = n;
      np->type = Latch;
      np->skew = 0;
      np->phase = -1;
      np->level = -INF;
      np->nai = 0;
      np->mindel = 0;
      np->maxdel = 0;
      np->minto =INF;
      np->minti = INF;
		
      validnode[n] = 1;
      sprintf(name,"FF_%s_%d",npin->name,i);
      strcpy(np->name,name);
	
      q1 = (Q *)MALLOC(1,sizeof(Q));
      q1->el = n;
      q1->next = latches;
      latches = q1;
      nolatches++;
    }

  outlist[0].latch_id = in;
  outlist[0].nofanouts = 0;
  outlist[0].fanout_list = NULL;
	
  /** do all fanuts of in and put them in right place**/
  for(w1= npin->fout;w1 != NULL; w1 = w1->next)
    {

      n1 = w1->el;
      wt = w1->wt;
      np1 = nodes[n1];
      for(w2 = np1->fin;w2 != NULL;w2 = w2->next) /* all fanins  TRICY what if 2 edges with diff wts from a to b*/
	{
	  if( (w2->el == in)&&(w2->wt == wt))
	    {
	      w2->el = outlist[wt].latch_id;
	      w2->wt = 0;
	      break;
	    }
	}

      w3 = (W *)MALLOC(1,sizeof(W));
      w3->list = NULL;
      w3->el = n1;
      w3->wt = 0;
      w3->next = outlist[wt].fanout_list;
      outlist[wt].fanout_list = w3;
      outlist[wt].nofanouts++;
    }

  /***** now move from outlist to correct place ***/
  free_W(npin->fout);
  npin->fout = outlist[0].fanout_list;
  npin->nofout = outlist[0].nofanouts;

  for(i=1;i<=maxw;i++)
    {
      n = outlist[i].latch_id;
      n1 = outlist[i-1].latch_id;
      np = nodes[n];
      np1 = nodes[n1];

      np->fout = outlist[i].fanout_list;
      outlist[i].fanout_list =NULL;
      np->nofout = outlist[i].nofanouts;

      /*** add prev to its fanin **/
      w3 = (W *)MALLOC(1,sizeof(W));
      w3->list = NULL;
      w3->next = NULL;
      w3->el = n1;
      w3->wt = 0;
      np->fin = w3;
      np->nofin = 1;

      /*** add this to prev's fanout *******/
      w3 = (W *)MALLOC(1,sizeof(W));
      w3->list = NULL;
      w3->el = n;
      w3->wt = 0;
      w3->next = np1->fout;
      np1->fout = w3;
      np1->nofout++;
    }
  free(outlist);
  return(maxw);
}
    
/************************************
 to be tested, not in use
For the alternative "merged latch" approach above
************************************/
void graph2mergedckt()
{
  int i,n,t;
   
  nolatches =0;
  for(i=0;i<nopin;i++)
    { 
      n =primary_input[i];
      t = add_merged_latches_at_fanout(n);
    }
  for(i=0;i<nogates;i++)
    { 
      n =gates[i];
      t = add_merged_latches_at_fanout(n);
    }
}	
/***********EOF********/
