/*******************************************************************
*  MINARET (for edge-triggered circuits)
*
*  BY
*
*  NARESH MAHESHWARI AND SACHIN S. SAPATNEKAR
*
*  Copyright 1998 Iowa State University Research Foundation, Inc.
*  All Rights Reserved
*
*  Source Code for retiming edge-triggered circuits
*
*  ASTRA:   Refer to paper in DAC 95 by Deokar & Sapatnekar
*                       and IEEE Tran on CAD 10/1998
*  Minaret: Refer to paper in DAC 97 by Maheshwari & Sapatnekar
*                       and IEEE Tran on VLSI 10/1998
*
*  Contact Address: sachin@ece.umn.edu
*
*        Availiable on as is basis, with no support
*******************************************************************/
/*******************************
FILE heap.c

Heap routines for period constraint generation.
Sizes of heap should be handled carefully, since they
can waste a lot of memory.

An array is used for heap with algorithms taken from Cormen,Leiserson 
and Revest's algorithm book.

******************************/

#include "include.h"

void init_heaps(int, int);
void free_heaps();
void heap_insert(int, float, int);
INT_FLT get_max(int );
int get_min_index();
void print_heap(int);
void reset_heaps();

/*******************************
GLOBAL VARS
******************************/

/************************************ 
all heap array start from 1 (not 0) so that 2*i is defined,
(needed to get parent/child realtionship)

As many as #FF heaps may be required, not just one heap.
Having all heaps as static arrays will require 
|G|*|F| (|G| = #gates, |F| = #FFs) memory. 
Using full dynamic memory allocation will require linked lists.
So I allocate memory for the whole array (for one heap) of size |G| at a time.
But allocation is done only as  needed, i.e., if only K heaps are actually needed
only K allocation 9each of size |G| is made).
"no_heaps" is the # of heaps currently allocated memory,
memory freed only in free_heap, so no_heaps is never decreased.
These are kept global, since period constraints are generated from 
one source at a time.
So initilization and free of heaps is done only once and not 
for every source gate, the heaps are reset for every source without 
freeing the memory, will do same with
a_row, b_row in wd.c
 
IMPORTANT: heap size can grow to be > nogates as gates can come in multiple times 
in the heap, don't know a good bound on the size of heap.
********************************/

typedef struct
{
  int size;
  INT_FLT *set;
} HEAPTYPE;

int Max_no_heaps;
int no_heaps; /* current no of heaps */
int Max_size;

HEAPTYPE *heaps;

extern int dbwd;
void heapify(int index,int element);


/**********************************
init_heaps(int max_no_of_heaps, int max)

Allocates memory for array pointers, not for individual heaps.
******************************/
void init_heaps(int max_no_of_heaps, int max)
{
  int i;
  heaps = (HEAPTYPE *)CALLOC((max_no_of_heaps+1),sizeof(HEAPTYPE));
  Max_no_heaps = max_no_of_heaps;
  for(i=0;i<max_no_of_heaps;i++)
    { 
      heaps[i].size = 0;
    }
  heaps[0].set = (INT_FLT *)MALLOC((max+1),sizeof(INT_FLT));
  no_heaps = 1;
  Max_size = max;

  /*printf(" initilized %d heaps of size %d\n",no_heaps,max);*/
}

/*******************************************
reset_heaps()
To be used before every new row of WD matrixes is generated.
Notice that the heap entry are not overwritten, they are just
invalidated by making "heap.size = 0".
******************************************/
void reset_heaps()
{
  int i;
  for(i=0;i<no_heaps;i++)
    { 
      heaps[i].size = 0;
    }
}

/********************************************
free_heaps()

*****************************************/
void free_heaps()
{
  int i;
  for(i=0;i<no_heaps;i++)
    { 
      if(heaps[i].size != 0)printf("\n WARNING {free_heaps} Heap %d not empty has %d elements\n",i,heaps[i].size);
      free(heaps[i].set);
    }
  free(heaps);
}

/**********************************************
get_min_index()

returns -1 if all heaps are empty 
***************************************/
int get_min_index()
{
  int i =0;
  while((heaps[i].size == 0)&&(i < no_heaps ))i++;
  if(i == no_heaps) i= -1;
  return (i);
}

/*****************************************************
heapify(int index,int element)

relocate elements in the array to preserve heap property
*******************************************/

void heapify(int index,int element)
{
  int left, right;
  int largest;
  INT_FLT temp;
  left = element << 1;
  right = left | 1;
  /* left = 2*element and right = 2*element + 1
     printf(" el %d r %d l %d\n",element,right,left); */


  if( (left <= heaps[index].size) && (heaps[index].set[left].value > heaps[index].set[element].value)) largest = left;
  else largest = element;

  if( (right <= heaps[index].size) && (heaps[index].set[right].value > heaps[index].set[largest].value)) largest =
													   right;

  if(0){
    printf("el %d largest = [%d]id %d v%.1f \n",element,largest,heaps[index].set[largest].id,heaps[index].set[largest].value);
    print_heap(0);
    printf("------\n");
  }
  if(largest != element)
    {
      temp.value = heaps[index].set[largest].value;
      temp.id = heaps[index].set[largest].id;

      heaps[index].set[largest].value = heaps[index].set[element].value;
      heaps[index].set[largest].id = heaps[index].set[element].id;
      heaps[index].set[element].value = temp.value;
      heaps[index].set[element].id = temp.id;

      heapify(index,largest);
    }
}



/*****************************************************
heap_insert(int index, float value, int id)

Insert <"value,id"> in heap["index"]
****************************************************/
void heap_insert(int index, float value, int id)
{
  int i,p;

  if(dbwd)printf("$$$ inserting in heap %d val %f id %d\n",index,value,id);

  if (index >=Max_no_heaps){
    printf("\nERR{heap_insert} index %d Max_no_heaps %d out of bound for value %f id %d \n",index,Max_no_heaps,
	   value,id);
    exit(0);
  }

  /******** if heap[index] has not been allocated memory do so now ***/
	
  if(index >=no_heaps){
    for(i=no_heaps;i<=index;i++){
      heaps[i].set = (INT_FLT *)MALLOC((Max_size+1),sizeof(INT_FLT));
      heaps[i].size =0;
    }
    no_heaps = index + 1;
  }

  if(heaps[index].size >= Max_size){
    printf("\n ERR{heap_insert} Heaps[%d] out of bound, max %d size %d\n",index,Max_size,heaps[index].size);
    print_heap(index);
    exit(0);
  }


  heaps[index].size ++;
  i = heaps[index].size;
  p = i >> 1;
  while( (i >1) && (heaps[index].set[p].value < value))
    {
      heaps[index].set[i].value = heaps[index].set[p].value;
      heaps[index].set[i].id = heaps[index].set[p].id;
      i = p;
      p = i >> 1;
    }
  heaps[index].set[i].value = value;
  heaps[index].set[i].id = id;
}

/*******************************
get_max(int index)

get the maximum tupe from heap["index"], and preserve the heap property
***************************************/
INT_FLT get_max(int index)
{
  INT_FLT res;
  if(heaps[index].size < 1){
    printf("\n ERR{get_max} heap[%d] underflow size %d\n",index,heaps[index].size); 
    exit(0);
  }

  res.id =  heaps[index].set[1].id;
  res.value =  heaps[index].set[1].value;

  heaps[index].set[1].value = heaps[index].set[heaps[index].size].value;
  heaps[index].set[1].id = heaps[index].set[heaps[index].size].id;
  heaps[index].size--;

  heapify(index,1); /*preserve the heap property*/
  return(res);
}

/***********************************
print_heap(int index)

prints heap["index"]; for debugging
***********************************/
void print_heap(int index)
{
  int i;
  printf(" Heap[%d] size %d is  ",index,heaps[index].size);
  for(i=1;i<=heaps[index].size;i++){
    printf(" [%d]id %d %.2f  ",i,heaps[index].set[i].id,heaps[index].set[i].value);/*if(i%5 == 0)*/
    printf("\n");
  }
  printf("\n");

}


/*****************EOF******************/
