/*******************************************************************
*  MINARET (for edge-triggered circuits)
*
*  BY
*
*  NARESH MAHESHWARI AND SACHIN S. SAPATNEKAR
*
*  Copyright 1998 Iowa State University Research Foundation, Inc.
*  All Rights Reserved
*
*  Source Code for retiming edge-triggered circuits
*
*  ASTRA:   Refer to paper in DAC 95 by Deokar & Sapatnekar
*                       and IEEE Tran on CAD 10/1998
*  Minaret: Refer to paper in DAC 97 by Maheshwari & Sapatnekar
*                       and IEEE Tran on VLSI 10/1998
*
*  Contact Address: sachin@ece.umn.edu
*
*        Availiable on as is basis, with no support
*******************************************************************/

/************************
FILE main.c
************************/

#include<string.h>
#include<stdlib.h>
#include<time.h>

#include "include.h"
#include "main.h"


#ifdef DEC
#include <sys/time.h>
#include <sys/resource.h>
#endif

/*******************************
 FUNCTIONS IN THIS FILE
******************************/
void min_area( float, float);
void min_period(float init_clock, float target_clk);

/*******************************
EXTERNAL VARIABLES
******************************/

extern int No_S_arcs;
extern int No_S_nodes;

extern int no_heaps;
extern int Latch_Remove_Counter;
extern int Latch_Add_Counter;
extern float P_bound;
extern float M_index;
extern int Max_UL;
extern int No_node;
extern int no_realloc;
extern int Arc_Array_Size;
extern int max_wd;

/*******************************
GLOBAL VARS
******************************/

int temp_counter=0;
FILE *yyin;
int mincost =0;
FILE *fpmin;
int MFF,init_MFF; /** number of FFs in the circuit taking FF sharing into account **/
int Red_Mode = REDUCE; /* if REDUCE use bounds (minaret approach) If SHENOY then do not use bounds */
int Use_Tran=0;
FILE *fprest1;

char  cktfile[100], resfile[50], tp_st[20],finalckt[100];

int Leak =0;  /** used to leave a memory leak in yacc.y, program runs faster this way ! */
int FAST_BELL =1;  /** used to select if FF's will be converted to single output before performing Bellman-Ford or after ****/

float tread;

/**************************************************************
main

interprets the different arguments, opens needed files and calls appropiate functions to perform retiming
valid usage is 
minaret <input file name> <output file name> <area|period> [Target clock]
area => minarea retiming using Minaret 
          if target clock not given then perform uncosntrained minarea retiming 
	  else perform constrained minarea retiming
period => minperiod retiming using ASTRA 
          if target clock not given then find optimal clock period 
	  else find retiming for given period without area consideration
	  
*************************************************************/

main(int argc,char **argv)
{

  FILE  *fpckt, *fpout;
  float init_clock1,init_clock2;
  float clk=0;  /* target clock; set to -1 if target clock is not given **/
  int do_minarea,write_ckt;
  float tbegin,tend;
  char fexten[10];
	
  printf(" START ***************************************\n");

  init_ckt_stuff(); /* imp initilizations */

  if(argc ==4)
    {clk = -1; }
  else if(argc ==5)
    {clk = atof(argv[4]);}
  else {printf(" \n invalid number of arguments \n usage retime <input file name> <output file name> <area|period> [Target clock] \n");
  exit(0);
  }
  /*** Read in valid arguments */
  strcpy(Ckt_Name,argv[1]); /* Ckt_Name is a global variable **/
  strcpy(cktfile,argv[1]);  /* input file name */
  strcpy(finalckt,argv[2]); /* output file name */
	
  if(!strcmp(argv[3],"area"))
    {do_minarea = 1;}
  else if(!strcmp(argv[3],"period"))
    {do_minarea = 0;}
  else {printf(" invalid argument %s \n valid options are <area> and <period> only \n\n ",argv[3]); exit(0);}  
   
  fperr  = fopen("Retime.log" ,"a");  /** put results and error messages in a log file */
  if(fperr ==NULL) {
    printf("cannot open file Minaret.log \n");
    fflush(stdout);
    exit(0);
  }

  fpres = fperr;

  Red_Mode = REDUCE;

	
#ifdef DEC
  getrusage(who,rusage);
  tbegin = rusage->ru_utime.tv_sec+rusage->ru_utime.tv_usec*1.e-6;
#else	
  tbegin = clock();
#endif
  /******** READING CIRCUIT ****/
	
  fpckt = fopen(cktfile ,"r");
  if(fpckt ==NULL) {
    printf("cannot open file %s\n",cktfile);
    fflush(stdout);
    exit(0);
  }

  yyin =fpckt;
  yyparse();
  fclose(fpckt);

#ifdef DEC
  getrusage(who,rusage);
  tend = rusage->ru_utime.tv_sec+rusage->ru_utime.tv_usec*1.e-6;
  tspent = tend - tbegin;
#else	
  tend = clock();
  tread = (tend - tbegin) /CLOCKS_PER_SEC;
#endif
	
  if(do_minarea == 1)
    {
      if(clk < 0) printf(" Performing unconstrained minarea retiming on circuit %s \n",Ckt_Name);
      else printf(" Performing constrained minarea retiming on circuit %s \n \t with target clock period %.2f \n ",Ckt_Name,clk);
    }
  else
    {
      if(clk < 0) printf(" Performing minperiod retiming on circuit %s \n ",Ckt_Name);
      else printf(" Performing given period retiming on circuit %s \n \t with target clock period %.2f \n",Ckt_Name,clk);
    }

  fprintf(fpres,"\n\n Running Ckt %s do_minarea %d clk %.2f",Ckt_Name,do_minarea,clk);

  /** set delays for each gate, use 1.0 for unit delays **/
  set_delays(1.0);

  /**** find the initial clock period **/
  init_clock1 = 100;
  init_clock1 = forwardpert(0);
  init_clock2 = reversepert(0);
  if(init_clock1 != init_clock2){
    printf("ERR for clk %.2f rev %.2f\n",init_clock1,init_clock2);
    fprintf(fpres,"ERR for clk %.2f rev %.2f\n",init_clock1,init_clock2);
    exit(-1);
  }

  printf(" Current/Initial clock period is %.2f \n",init_clock1);

  fprintf(fpres," Clk Init %.1f Spec %.1f RM %d \n",init_clock1,clk,Red_Mode);


  reset_ckt();
  check_nodeid();

  /** perform retiming **/

  if(do_minarea)min_area(init_clock1,clk);
  else min_period(init_clock1,clk);
	
  /** open output file to write circuit **/
  fpout = fopen(finalckt,"w");
  if(fpout ==NULL) {
    printf("cannot open file %s\n",finalckt);
    fflush(stdout);
    exit(0);
  }

  remove_artificial_gates();

  if(0)write_benchmark(fpout);  /*** for proper MCNC benchmark ***/
  if(1)write_netlist(fpout);  /*for modified MCNC format **/


  fclose(fperr);
  free_ckt();
  printf(" DONE ********************************************************************\n");
         
}

/******************************************
***************************************************
performs  constrined/unconstrained  minarea retiming  using Minaret
(see Maheshwari & Sapatnekar, Tran on VLSI 3/98 )
init_clock is the initial clock period (only used for reporting results)
clk is the target clock period;
if clk < 0 then perform uncosntrained minarea retiming

Formulate the minarea retiming problem as a LP and solve it using network simplex algorithm
***********************************************
********************************************/

void min_area(float init_clock, float clk)
{

  int i,n,dbl =0;
  float got_clock1,got_clock2;
  int obj;  /* objective function of the LP **/
	
  NODETYPE *np;
  int init_nolatches;
  int reduced_var;
  int Max_w;
  int no_mf;

  /* variables for measuring CPU time */
  float tbegin,tend,tspent;
  float time_bounds, time_wd, time_soln,time_exec;

  int  to_mult;
  int iter;
  int UNconstrained=0;

#ifdef DEC
  int who =0;
  struct rusage *rusage;
  
  who =0;
  rusage = (struct rusage *) malloc(sizeof(struct rusage));
#endif


  init_nolatches = nolatches;

  if(clk <0){  /*** need to perform unconstrained minarea **/
    UNconstrained =1; 
    clk = 20000;  /** a large clock period for bound calculation in case of uncosntrained minarea retiming; get_bounds can be modified/optimized to generate bounds without using clk **/
  }
	
  /**********************************
    decide if FFs hsould be made single fanout here or later
    *********************************/
  if(!FAST_BELL)
    {to_mult = make_single_fanout(1);}

  /*******************************************
    GET BOUNDS ON GATES
    ******************************************/

#ifdef DEC
  getrusage(who,rusage);
  tbegin = rusage->ru_utime.tv_sec+rusage->ru_utime.tv_usec*1.e-6;
#else	
  tbegin = clock();
#endif
  if(Red_Mode == REDUCE )reduced_var = get_bounds(clk);
  if(Red_Mode == SHENOY)
    {
      for(i=0;i<nogates;i++)
	{  
	  n= gates[i];
	  np = nodes[n];
	  np->bounds[LOWER] = -INF;
	  np->bounds[UPPER] = INF;
	  np->IsFixed = 0;
	}
      ckt2graph();
      remove_pio();
      free_all_latches();
      free_saved_latches();
    }

#ifdef DEC
  getrusage(who,rusage);
  tend = rusage->ru_utime.tv_sec+rusage->ru_utime.tv_usec*1.e-6;
  tspent = tend - tbegin;
#else	
  tend = clock();
  tspent = (tend - tbegin) /CLOCKS_PER_SEC;
#endif
  time_bounds = tspent;


  /*******************************************
    bounds may give all gates as fixed in which case there is no need to solve the LP;
    need to modify the code to output a valid circuit in this case
    ******************************************/
  if(reduced_var == nogates){
    printf(" \n\n DONE no need to solve the problem all gates are fixed \n");
    exit(0);
  }


  init_MFF = count_latches_in_graph();
  Max_w = init_MFF +1;  /*** used for size of heap in generating the constraints **/

  /*******************************************
    Generate the LP
    ******************************************/
#ifdef DEC
  getrusage(who,rusage);
  tbegin = rusage->ru_utime.tv_sec+rusage->ru_utime.tv_usec*1.e-6;
#else	
  tbegin = clock();
#endif
	
  start_simplex_network(Red_Mode); /* starts the simplex graph and puts circuit (+ mirror) constrsints */

#ifdef DEBUG
  /*** just printing the bounds on gates and mirror vertices **/
  if(0)for(i=0;i<nogates;i++){
    n= gates[i];
    np = nodes[n];
    if(np->IsFixed);
    printf("gate [%d]%s  Bounds L %d U %d  \n",n,np->name,np->bounds[LOWER],np->bounds[UPPER]);
    printf("mirror [%d]%s Ism %d Bounds L %d U %d  \n",n,np->name,np->Ism,np->Lm,np->Um);

  }
#endif

  store_moves();

  /*** generate the period constraints only is constrained minarea retiming is to be performed */
  if(!UNconstrained){
    generate_period_cons(clk,Red_Mode,Max_w);
  }
	
#ifdef DEC
  getrusage(who,rusage);
  tend = rusage->ru_utime.tv_sec+rusage->ru_utime.tv_usec*1.e-6;
  tspent = tend - tbegin;
#else	
  tend = clock();
  tspent = (tend - tbegin) /CLOCKS_PER_SEC;
#endif
  time_wd = tspent;


  printf(" Size of LP is # Variables %d;  # Constraints %d \n",No_S_nodes,No_S_arcs);

  /*******************************************
    Solve the LP
    ******************************************/

#ifdef DEC
  getrusage(who,rusage);
  tbegin = rusage->ru_utime.tv_sec+rusage->ru_utime.tv_usec*1.e-6;
#else	
  tbegin = clock();
#endif

  make_simplex_tree(Red_Mode);
	
  iter = solve_simplex();

#ifdef DEC
  getrusage(who,rusage);
  tend = rusage->ru_utime.tv_sec+rusage->ru_utime.tv_usec*1.e-6;
  tspent = tend - tbegin;
#else	
  tend = clock();
  tspent = (tend - tbegin) /CLOCKS_PER_SEC;
#endif
  time_soln = tspent;



  /*******************************************
    Read Solution
    ******************************************/

  obj = read_simplex(Red_Mode);

  /*** correct the optimal value by adding the constant part to the objective function */
  OBJ_VALUE = OBJ_VALUE + obj;

  /*** need to have all the Primary input outputs restored before applying retiming **/
  restore_pio();

  /** do the actual retiming **/
  apply_retiming();

  MFF = count_latches_in_graph(MERGE);

  /*** Check if final #FF is as expected **/
  if(init_MFF - OBJ_VALUE != MFF){printf("\n\nERR obj init latch %d OBJ val %d current is %d \n", init_MFF,OBJ_VALUE,MFF);exit(-1);}

	
#ifdef DEBUG
  if(0)print_modified_graph(0);
  if(0)print_reverse_modified_graph(0);
#endif
	
  /******************************** 
    Convert the graph to circuit
    **********************/
  graph2mergedckt();

  reset_ckt();

  /**** check final clock period **/

  got_clock1 = forwardpert(0);
  got_clock2 = reversepert(0);
  if(got_clock1 != got_clock2){
    printf("ERR for clk %.2f rev %.2f\n",got_clock1,got_clock2);
    exit(0);
  }
	
  if(got_clock1 > clk){
    printf("ERR  for clk got %.2f spec %.2f\n",got_clock1,clk);
    exit(-1);
  }

  merge_latches();
  no_mf = nolatches;

  if(no_mf != MFF){
    printf(" ERR MFF %d but latches after merge %d \n",MFF,no_mf);
    fprintf(fpres,"$$$$ ERR MFF %d but latches after merge %d \n",MFF,no_mf);
  }
	
  fprintf(fpres," after merge latch %d init %d \n",nolatches,init_MFF);


  /*******************************************
    PRINT RESULTS
    ******************************************/

  time_exec = time_bounds + time_wd + time_soln;

  if(0)printf(" %s spec clk %.1f, INIT no latch %d clk %.1f FINAL no latch Merged %d  clk %.1f\n",Ckt_Name,clk,init_nolatches,init_clock,no_mf,got_clock1);
  printf(" Final number of FFs = %d \n" ,MFF);
  printf("\n Execution Time Summary in Seconds \n Obtain Bounds \t\t %.2f \n Generate LP \t\t %.2f  \n Solve LP \t\t %.2f\n TOTAL \t\t\t %.2f \n\n",time_bounds,time_wd,time_soln,time_exec);

 fprintf(fpres,"%s & %.2fs & %.2fs & %.2fs & %.2fs & %d & %d \\\\ \\hline\n",Ckt_Name,time_bounds,time_wd,time_soln,time_exec,No_S_nodes,No_S_arcs);
       
#if 0
  /*** various print statements to get table of results ***/

  fprintf(fpres,"R %s & %.1f/%.1f & %.2f ",Ckt_Name,got_clock1,clk,(float)100*No_S_arcs/Arc_Array_Size);
  fprintf(fpres,"& %.1f & %.2f & %d & %d & %d & %d ",P_bound,M_index,Max_UL,No_node,MFF,no_realloc);
  fprintf(fpres,"& %.2fs & %.2fs & %.2fs & %.2fs & %d & %d \\\\ \\hline\n",time_bounds,time_wd,time_soln,time_exec,No_S_nodes,No_S_arcs);

  fprintf(fpres,"T1 %s &  %d & %.1f &   & %d & %.2fs \\\\ \\hline \n ",Ckt_Name,nogates,clk,MFF,time_exec);
  fprintf(fpres,"C %s &  %d & %.1f & %d & %d & & %d & %d %  \\\\ \\hline \n ",Ckt_Name,nogates,clk,init_MFF,MFF,init_Cuts,final_Cuts);
  fprintf(fpres,"TAD %s & %.1f &   & %d & %.2f & %.2f & %.2fs\\\\ \\hline \n ",Ckt_Name,clk,MFF,M_index,P_bound,time_exec);
  fprintf(fpres,"T2 %s & %.1f & %.2f & %.2f & %d & %d  &  & \\\\ \\hline \n ",Ckt_Name,clk,M_index,P_bound,No_S_nodes,No_S_arcs);

#endif

}

/************************************************************************
************************************
performs minperiod/given period retiming using ASTRA 
(see Sapatnekar and Deokar, Tran on CAD 10/96)

init_clock is the initial clock period (only used for reporting results)
clk is the target clock period;
if clk < 0 then perform minperiod retiming

**************************************************************************/
void min_period(float init_clock, float target_clk)
{
  int i,n,dbl =0,t1,final_nolatches;
  float got_clock1,got_clock2,opt_clock,final_clock;
  float f1,cycle_period;

  /* variables for CPU time measurment */
  float tbegin,tend,tspent;
  float time_a, time_b, time_t;

  db_bin = 0; db_opt =0;        
	
  /***************************************
    Phase A
    ************************************/    
#ifdef DEC
  getrusage(who,rusage);
  tbegin = rusage->ru_utime.tv_sec+rusage->ru_utime.tv_usec*1.e-6;
#else	
  tbegin = clock();
#endif

  /*** make the constriant graph for phase A ****/
  all_pert();
  make_reverse_graph();

  /** find the optimal clock period **/
  if(target_clk <= 0)
    {
      printf(" Searching for optimal clock period \n");
      opt_clock = opt_period();
      normalise_skews(FORWARD,0);
        
      printf(" Optimal Clock period found as %.2f;  retiming circuit now ... \n",opt_clock);
    }
  else
    {cycle_period = bellford(target_clk,0,FORWARD);
    normalise_skews(FORWARD,0);
    if(cycle_period > 0)
      { printf("Target clock %.2f **NOT** feasible for circuit %s \n Try larger clock period or minperiod retiming \n",target_clk,Ckt_Name);exit(0);}
    else
      {printf(" Clock period %.2f is feasible; retiming circuit now ... \n",target_clk);}
    }

#ifdef DEC
  getrusage(who,rusage);
  tend = rusage->ru_utime.tv_sec+rusage->ru_utime.tv_usec*1.e-6;
  tspent = tend - tbegin;
#else	
  tend = clock();
  tspent = (tend - tbegin) /CLOCKS_PER_SEC;
#endif
  time_a = tspent;

  /***************************************
    Phase B
    ************************************/    

#ifdef DEC
  getrusage(who,rusage);
  tbegin = rusage->ru_utime.tv_sec+rusage->ru_utime.tv_usec*1.e-6;
#else	
  tbegin = clock();
#endif
  f1 = forwardpert(1);
  f1 = reversepert(1);

  /** Convert circuit to graph **/
  ckt2graph();
  remove_pio();
  free_all_latches();
  restore_ckt_graph(0); /** transfer skew to the circuit graph from the constraint graph **/

  /**** Do actual retiming **/
  t1 = retime(opt_clock,1,1,UPPER,1);  

  /*** Convert back to circuit **/
  free_saved_latches();
  restore_pio();
  final_nolatches = count_latches_in_graph(MERGE);
  graph2mergedckt();
  reset_ckt();

#ifdef DEC
  getrusage(who,rusage);
  tend = rusage->ru_utime.tv_sec+rusage->ru_utime.tv_usec*1.e-6;
  tspent = tend - tbegin;
#else	
  tend = clock();
  tspent = (tend - tbegin) /CLOCKS_PER_SEC;
#endif
  time_b = tspent;
  time_t = time_a + time_b;

  /**** Find and check the final clock period **/
  got_clock1 = 100;
  got_clock1 = forwardpert(0);
  got_clock2 = reversepert(0);
  if(got_clock1 != got_clock2){
    printf("ERR for clk %.2f rev %.2f\n",got_clock1,got_clock2);
    fprintf(fpres,"ERR for clk %.2f rev %.2f\n",got_clock1,got_clock2);
    exit(-1);
  }

  printf(" Retimed the circuit successfully; final clock period is %.2f; # FF %d\n",got_clock1,final_nolatches);

  if(target_clk < 0)printf(" Optimal clock period with skews was %.2f \n",opt_clock);

  fprintf(fpres," Clk Init %.1f Skew Clock %f Final Clock %f \n",init_clock,opt_clock,got_clock1);

  fprintf(fpres," Ckt %s G %d Tread %.2f TA %.2f TB %.2f TT %.2f \n",Ckt_Name,nogates,tread,time_a,time_b,time_t);
 
 fprintf(fpres," T %s & %d & %.2f & %d & %.2f & %.2f & %.2f EOF\n",Ckt_Name,nogates,got_clock1,final_nolatches,time_a,time_b,time_t);

  printf("\n Execution Time Summary in Seconds\n Phase A \t\t %.2f \n Phase B \t\t %.2f  \n  TOTAL \t\t %.2f \n\n",time_a,time_b,time_t);

  /**** end of minperiod ***/
}
