/*******************************************************************
*  MINARET (for edge-triggered circuits)
*
*  BY
*
*  NARESH MAHESHWARI AND SACHIN S. SAPATNEKAR
*
*  Copyright 1998 Iowa State University Research Foundation, Inc.
*  All Rights Reserved
*
*  Source Code for retiming edge-triggered circuits
*
*  ASTRA:   Refer to paper in DAC 95 by Deokar & Sapatnekar
*                       and IEEE Tran on CAD 10/1998
*  Minaret: Refer to paper in DAC 97 by Maheshwari & Sapatnekar
*                       and IEEE Tran on VLSI 10/1998
*
*  Contact Address: sachin@ece.umn.edu
*
*        Availiable on as is basis, with no support
*******************************************************************/
/************************
FILE simplex.c

contains routines for solving mincost network flow problem using
network simplex algorithm (from books by Ahuja, and Bazaraa).
Using simplex algorithm from Bazaraa's book with some simplification.
************************/


#include<string.h>
#include<stdlib.h>
#include<time.h>

#include "include.h"

extern int db_sim0;
extern int db_sim1;
extern int db_sim2;



/*******************************
EXTERNAL FUNCTIONS
******************************/
void apply_retiming();
void ckt2graph();
void graph2ckt();
void remove_pio();
void restore_pio();
void check_nodes_S_Arcs();
/*******************************
  GLOBAL VARS
  ******************************/

S_ARC *S_Arcs;
S_NODE *S_nodes;

S_ARC *New_S_Arcs;

int Arc_Array_Size;
int No_S_nodes;
int No_S_arcs;

int Root;
int Min_cost;

int Start_arc = 0;
/*******************************
solve simplex

solves the network flow problem and returns number of iteration needed for the solution.
the solution is kept in the simplex data structure.
need to call make_simplex_tree() before this function
******************************/

int  solve_simplex()
{
  int Entering_arc,y,z;
  int red;
  int p,q,u,v,k;
  int delta;
  int From,To,Parent_end,Child_end,tmp,enter_orient;
  int iter = 0;

#ifdef CHECK  
  check_nodes_S_Arcs();
#endif

  Entering_arc = get_entering_arc();

  while(Entering_arc != -1)
    {
#ifdef CHECK  
      if((Entering_arc <0)|| (Entering_arc  >= No_S_arcs)){
	printf(" ERR in solve entering arc  %d \n",Entering_arc );
	exit(0);
      }
#endif

#ifdef DEBUG
      if(db_sim1)printf(" Mincost %d entering arc %d from %d to %d cost %d red_cost %d \n",Min_cost,Entering_arc,S_Arcs[Entering_arc].from,
			S_Arcs[Entering_arc].to,S_Arcs[Entering_arc].cost,S_Arcs[Entering_arc].red_cost);
#endif

      From = S_Arcs[Entering_arc].from;
      To = S_Arcs[Entering_arc].to;
      red = find_cycle(From,To,&tmp, &k,  &u, &v);
      Min_cost -= S_Arcs[Entering_arc].red_cost * red;

#ifdef DEBUG
      if(db_sim1)printf(" red %d Min_cost %d \n",red,Min_cost);
#endif

      if(tmp == To) { 
	Parent_end = From; 
	Child_end = To; 
	enter_orient = -1; 
	delta = S_Arcs[Entering_arc].red_cost;
      }
      else if(tmp == From) {
	Parent_end = To; 
	Child_end = From;
	enter_orient = 1; 
	delta = -1* S_Arcs[Entering_arc].red_cost;
      }
      else { 
	printf("ERR find cycle gave a tmp/tree of %d while From was %d To %d \n",tmp,From,To);
      }

#ifdef DEBUG
      if(db_sim1) printf(" Entering arc %d to %d leaving arc parent(u) %d child (v) %d \n",From,To,u,v);
      if(db_sim1) printf(" Parent_end %d Child_end %d \n",Parent_end,Child_end);
#endif

      update_stem(Parent_end,Child_end,u,v,red,enter_orient);

      update_dual_and_level(Parent_end,Child_end,v,delta);

      Entering_arc = get_entering_arc();
      iter++;

#ifdef CHECK  
      /*if(iter%1000 == 0)printf(" iter %d \t",iter);*/
#endif

    }
  if(0)printf(" OBJ %d \n",Min_cost);
  free(S_Arcs);
  return(iter);
}



/********************************************
 updates reduced cost of non basic S_Arcs;
also returns arc index of entering S_Arcs;
Now uses Start_arc to start looking from S_Arcs from there
IMP Start_arc has to be set to 0 by someone.
Uses a return heuristic of Ahuja's book: return first vaild arc rather
than the best arc; also start searching for arc next time from this location 
rather than from start.
***********************************************/
int get_entering_arc()
{
  int i;
  int max = -INF;
  int enter;

  for(i=Start_arc;i<No_S_arcs;i++)
    {
      S_Arcs[i].red_cost = S_nodes[S_Arcs[i].from].dual -  S_nodes[S_Arcs[i].to].dual - S_Arcs[i].cost;
      if(max < S_Arcs[i].red_cost)
	{
	  max = S_Arcs[i].red_cost; 
	  enter = i;
	  /****** RETURN HUERISTIC *****/
	  if(max>0) {
	    Start_arc = i;
	    return(enter);
	  }
	}
    }

  /*** Only to be used with return heuristic**/
  for(i=0;i<Start_arc;i++)
    {
      S_Arcs[i].red_cost = S_nodes[S_Arcs[i].from].dual -  S_nodes[S_Arcs[i].to].dual - S_Arcs[i].cost;
      if(max < S_Arcs[i].red_cost)
	{
	  max = S_Arcs[i].red_cost; 
	  enter = i;
	  if(max>0){
	    Start_arc = i;
	    return(enter);
	  }
	}
    }

  if(max <=0)enter = -1;

  return(enter);
}


/********************************************
returns maximum alowable flow in a cycle
***********************************************/
int find_cycle(int P, int q, int *tree, int *K, int *U, int *V)
{
  int k,l;
  int min = INF;
  int u,v,x;

  k = P;
  l = q;
  u = -1;
  v = -1;

#ifdef CHECK  
  if((P <0)|| (P >= No_S_nodes)){
    printf(" ERR in find_cycle P  %d \n",P);
    exit(0);
  }
  if((q <0)|| (q >= No_S_nodes)){
    printf(" ERR in find_cycle q  %d \n",q);
    exit(0);
  }
#endif

  while(k !=l)
    {
#ifdef DEBUG
      if(db_sim2)printf("loop in find_cycle P %d q %d k %d L %d u %d v %d min %d tree %d\n",P,q,k,l,u,v,min,*tree);
#endif

      if(S_nodes[k].level > S_nodes[l].level)
	{ 
	  if(S_nodes[k].orient == 1)
	    { 
	      if(min > S_nodes[k].flow){
		min =  S_nodes[k].flow; 
		v = k,u = S_nodes[k].parent; 
		*tree = P;
	      }
	    }
	  k = S_nodes[k].parent;
	}

      else if(S_nodes[k].level < S_nodes[l].level)
	{ 
	  if(S_nodes[l].orient == -1)
	    { 
	      if(min > S_nodes[l].flow){
		min =  S_nodes[l].flow; 
		v = l,u = S_nodes[l].parent;
		*tree = q;
	      }
	    }
	  l = S_nodes[l].parent;
	}
      else
	{
	  if(S_nodes[k].orient == 1)
	    { 
	      if(min > S_nodes[k].flow){
		min =  S_nodes[k].flow; 
		v = k,u = S_nodes[k].parent;
		*tree = P;
	      }
	    }
	  if(S_nodes[l].orient == -1)
	    { 
	      if(min > S_nodes[l].flow){
		min =  S_nodes[l].flow; 
		v = l,u = S_nodes[l].parent;
		*tree = q;
	      }
	    }

	  l = S_nodes[l].parent;
	  k = S_nodes[k].parent;

	}
    }

#ifdef DEBUG
  if(db_sim2)printf(" exit arc u %d v %d orient %d flow %d min %d P%d q%d K %d \n",u,v,S_nodes[v].orient,S_nodes[v].flow,
		    min,P,q,k);
#endif

  *K = k;
  *V = v;
  *U = u;

  /* flow assued to be from P to q updating flows on cycle */
  x= q;
  while (x != k)
    {
      S_nodes[x].flow +=  S_nodes[x].orient*min;
      x = S_nodes[x].parent;
    }

  x= P;
  while (x != k)
    {
      S_nodes[x].flow -=  S_nodes[x].orient*min;
      x = S_nodes[x].parent;
    }

  if(S_nodes[v].flow != 0){
    printf("ERR in find_cycle leaving arc u %d v %d flow still %d\n",u,v,S_nodes[v].flow);
  }
  if((u == -1)||(v== -1)){
    printf("ERR in find_cycle did't find leaving arc \n");
    exit(0);
  }
  return(min);
}


/********************************************
reverse parents on stem from Q to V 
and then updates flow & orient
ASSUMES V is on subtree of Q so P is made parent of Q;
***********************************************/
void update_stem(int P, int q,int U, int V, int flow, int orient)
{
  int t1,t2,t3;
  Q *q1;
  t1 = q;
  t2 = S_nodes[q].parent;

#ifdef CHECK  
  if((P <0)|| (P >= No_S_nodes)){
    printf(" ERR in find_cycle P  %d \n",P);
    exit(0);
  }
  if((q <0)|| (q >= No_S_nodes)){
    printf(" ERR in find_cycle q  %d \n",q);
    exit(0);
  }
  if((U <0)|| (U >= No_S_nodes)){
    printf(" ERR in find_cycle U  %d \n",U);
    exit(0);
  }
  if((V <0)|| (V >= No_S_nodes)){
    printf(" ERR in find_cycle V  %d \n",V);
    exit(0);
  }
#endif

#ifdef DEBUG
  if(db_sim0)printf("update_stem P %d q %d U %d V %d flow %d or %d\n",P,q,U,V,flow,orient);
  fflush(stdout);
#endif

  while (t1 != V)
    {
      t3 = S_nodes[t2].parent;
      S_nodes[t2].parent = t1;
      S_nodes[t2].child = delete_child(S_nodes[t2].child,t1);
      S_nodes[t2].nochild--;
      q1 = (Q *)MALLOC(1,sizeof(Q));
      q1->el = t2;
      q1->next = S_nodes[t1].child;
      S_nodes[t1].child = q1;
      S_nodes[t1].nochild++;

      t1 = t2;
      t2 = t3;
    }


  /* delete V from child of U */
  S_nodes[U].child = delete_child(S_nodes[U].child,V);
  S_nodes[U].nochild--;

  /* make P parent of Q */
  S_nodes[q].parent = P;
  q1 = (Q *)MALLOC(1,sizeof(Q));
  q1->el = q;
  q1->next = S_nodes[P].child;
  S_nodes[P].child = q1;
  S_nodes[P].nochild++;


  /* now update flow and orient, going up from v copy parent */

  t1 = V;
  while(t1 != q)
    {
      t2 = S_nodes[t1].parent;
      S_nodes[t1].flow = S_nodes[t2].flow;
      if(S_nodes[t2].orient == -1)S_nodes[t1].orient = 1;
      if(S_nodes[t2].orient == 1)S_nodes[t1].orient = -1;
      t1 = t2;
    }
  S_nodes[q].flow = flow;
  S_nodes[q].orient = orient;

}



/********************************************
add "change" to dual of all nodes in sub-tree of Q;
make level(Q) = levele(p) +1
and update level of node in subtree of Q
***********************************************/
void update_dual_and_level(int P,int q,int V,int change)
{
  int x,y,z,lv;
  Q *q1, *q2, *q3, *start, *end;

#ifdef CHECK  
  if((P <0)|| (P >= No_S_nodes)){
    printf(" ERR in find_cycle P  %d \n",P);
    exit(0);
  }
  if((q <0)|| (q >= No_S_nodes)){
    printf(" ERR in find_cycle q  %d \n",q);
    exit(0);
  }
  if((V <0)|| (V >= No_S_nodes)){
    printf(" ERR in find_cycle V  %d \n",V);
    exit(0);
  }
#endif



  start= (Q *)MALLOC(1,sizeof(Q));
  end = start;
  start->next = NULL;
  start->el = q;

  S_nodes[q].level = S_nodes[P].level +1;
  S_nodes[q].dual += change;
  while(start != NULL)
    {
      x = start->el;
      lv = S_nodes[x].level;
      for(q2 = S_nodes[x].child;q2 != NULL; q2 = q2->next)
	{
	  y = q2->el;
	  S_nodes[y].level = lv +1;
	  S_nodes[y].dual += change;
	  q1 = (Q *)MALLOC(1,sizeof(Q));
	  q1->el = y;
	  q1->next = NULL;
	  end->next = q1;
	  end = q1;
	}
      q3 = start;
      start = start->next;
      free(q3);
    }

}




/********************************************
delete only one occurance so that if the same child is
entered & deleted it's still there
***********************************************/

Q *delete_child(Q *list,int child)
{
  Q *q1, *prev, *t;


  /*** delete any "child"s in the start of Q **/
  if(list->el == child)
    {
      t = list->next; 
      free(list); 
      list = t;
    }

  /** list had 0 or only ns so return null ***/
  if(list == NULL)return(NULL);

  /* list-> el ! = n */
  prev = list;
  for(q1=list;q1!= NULL; q1= q1->next)
    {
      if(q1->el == child)
	{
	  prev->next = q1->next;
	  free(q1);
	  break;
	}
      prev = q1;
    }
  return(list);
}




/********************************************
initiates the simplex network, by adding nodes corresponding to
all gates and mirror vertex (only mobile gates for Minaret);
also adds the circuit and mirror constraints.

  uses:  phase to store id of S_node;
         Id_m to store id in S_node of mirror;
	 nai to store supply
	 level == 1 for variable; ==0 for fixed node;
	 nao == 1 if mirror is variable, 0 otherwise;
The coefficients in the objective function are calculated using a method that is different 
yet equivalent to the one in Leiserson Saxe's paper, this method is given in my PhD thesis.
Since each r(v) is actually r(v) - r(HOST); in our formulation so the  negative of the sum of coefficient of all variables (with a negative sign) is the coefficient of the host vertex.	 
***********************************************/
void start_simplex_network( int Red_Mode)
{

  int i,n,diff,n2,new_w;
  NODETYPE *np, *np2;
  W *w1;
  int Init_cost;
  int exp_arcs;/* expected no of arcs */
  int exp_nodes;/* expected no of variables */
  int constant_obj =0;  /* constant part of objective function for Minaret*/
  int host_obj=0;  /*coefficient of the host vertex */
  int cost;
  P_SIZE prob_size;
  int exp_m_arcs;  /* no of mirror arc expected (i.e, for a gate to a mirror) */


  No_S_nodes =0;
  No_S_arcs =0;
  exp_arcs =0;
  exp_nodes =0;
  constant_obj =0;
  host_obj =0;


  /****************************
    Mark nodes as variable & bound (including mirror nodes)
    using nai,nao,Ism,level;
    also get expected # variabless & # arcs
    *********************************** */
  for(i=0;i<(nogates-1);i++)
    {
      n = gates[i];
      np = nodes[n];
      np->phase = -1;
      np->Ism = -1;
      np->level = 0;
      np->nao = 0;
      if(np->nofout > 1)
	{
	  np->Ism = 1;
	  np->Maxm = 0;
	  np->Um = np->Lm = -100000;
	  exp_m_arcs = 0;
	  for(w1 = np->fout;w1 != NULL;w1 = w1->next)
	    {
	      n2 = w1->el;
	      np2 = nodes[n2];
	      if((Red_Mode == SHENOY)||(np2->IsFixed == 0))exp_m_arcs++;
	      if(np->Maxm < w1->wt)np->Maxm = w1->wt;
	      if(np->Lm < np2->bounds[LOWER] + w1->wt)np->Lm = np2->bounds[LOWER] + w1->wt;
	      if(np->Um < np2->bounds[UPPER]+ w1->wt)np->Um = np2->bounds[UPPER] + w1->wt;

	      if(w1->el == n){ 
		printf(" \n SELF LOOP IN A GATE WITH > 1 FOUT gate %d %s \n\n",n,np->name);
	      }
	    }/* for all fouts */
	  np->Um -= np->Maxm;
	  np->Lm -= np->Maxm;
	  
#ifdef MY_METHOD
	  if(np->Um == np->Lm)
#else
	    if((Red_Mode == REDUCE)&&(np->Um == np->Lm))
#endif
	      {
		constant_obj += -1*np->Um;
		np->nao = 0;
	      }
	    else
	      {
		np->nao = 1;
		host_obj -= -1;
		exp_nodes++;
		exp_arcs += exp_m_arcs;
	      }

	} /* if nofout > 1 */
      /* cost in merged mode is 1 - no of inputs with only one output (unshared FF at input) HOST to be treated as nofout = 1 */
      cost = 1;
      for(w1 = np->fin;w1 != NULL;w1 = w1->next)
	{
	  if((nodes[w1->el]->nofout == 1)||w1->el == HOST) cost--;
	}

      if((Red_Mode == REDUCE)&&(np->IsFixed)) /* is constant node NOT a var */
	{

	  constant_obj += cost*np->bounds[LOWER];
	  np->level = 0;
	  np->nai = 0;
	  np->phase = -1;
	}
      else
	{
	  exp_nodes++;
	  np->level = 1;
	  np->nai = cost;
	  exp_arcs += np->nofout; /* actually this snumber is larger than required, as some circuit arcs are dropped*/
	  host_obj -= cost;
	}

    }

  nodes[HOST]->level = 1;
  nodes[HOST]->nai = host_obj;
  nodes[HOST]->Ism = 0;
  exp_nodes++;
  exp_arcs += nodes[HOST]->nofout;

  OBJ_VALUE = constant_obj;



  if(Red_Mode == REDUCE)exp_arcs += 2*exp_nodes; /* arcs for lower and upper bounds */


  /*printf(" expected arcs %d nodes %d \n",exp_arcs,exp_nodes);*/

  /************** 
    counting # of expected arcs to simplify the memory allocation
    ********************/
  Arc_Array_Size = exp_arcs + 10;  /** just adding 10 to be extra safe**/
  S_Arcs = (S_ARC *)MALLOC(Arc_Array_Size,sizeof(S_ARC));
  if(S_Arcs == NULL){
    printf(" ERR out of mem did't give mem for %d arcs \n",exp_arcs);
    exit(0);
  }
  S_nodes = (S_NODE *)MALLOC((exp_nodes+10),sizeof(S_NODE)); /** just adding 10 to be extra safe**/
  if(S_nodes == NULL){
    printf(" ERR out of mem did't give mem for %d S_nodes \n",exp_nodes);
    exit(0);
  }


  /**** fill in the information for simplex nodes **/
  for(i= (nogates -1);i >=0;i--)
    {
      n = gates[i];
      np = nodes[n];

      if(np->level == 1)
	{
	  np->phase = No_S_nodes;
	  S_nodes[No_S_nodes].ckt_id = n;
	  S_nodes[No_S_nodes].supply = np->nai;
	  S_nodes[No_S_nodes].nochild = 0;
	  S_nodes[No_S_nodes].child = NULL;
	  No_S_nodes++;
	}
      if(np->nao == 1) /* mirror node */
	{
	  np->Id_m = No_S_nodes;
	  S_nodes[No_S_nodes].ckt_id = -1*(1+n);  /** this is the encoding for mirror nodes**/
	  S_nodes[No_S_nodes].supply = -1;
	  S_nodes[No_S_nodes].nochild = 0;
	  S_nodes[No_S_nodes].child = NULL;
	  No_S_nodes++;
	}
    }

  if(exp_nodes != No_S_nodes){
    printf("ERR {write_simplex} Expected  nodes %d got %d  \n",exp_nodes,No_S_nodes);
    exit(0);
  }


  Root = nodes[HOST]->phase;
  No_S_arcs =0;
  /* copy arcs */
  for(i=(nogates -1);i >=0;i--)
    {
      n = gates[i];
      np = nodes[n];

      if(np->level == 1) /* gate needs to be used */
	{
	  for(w1 = np->fout;w1 != NULL;w1 = w1->next)
	    {
	      n2 = w1->el;
	      np2 = nodes[n2];
	      if(np2->level == 1) /* only if fanout is a variable */
		{
		  new_w = np->bounds[UPPER] - np2->bounds[LOWER];
		  if(Red_Mode == SHENOY|| new_w > w1->wt) /* only if rule 2 does not apply */
		    {
		      S_Arcs[No_S_arcs].from = np->phase ;
		      S_Arcs[No_S_arcs].to = nodes[w1->el]->phase;
		      S_Arcs[No_S_arcs].cost = w1->wt;
		      No_S_arcs++;
		    }
		}
	    }
	  if((Red_Mode == REDUCE)&&(n != HOST)) /* do bounds */	
	    {
	      S_Arcs[No_S_arcs].from = np->phase;
	      S_Arcs[No_S_arcs].to = Root;
	      S_Arcs[No_S_arcs].cost = np->bounds[UPPER];
	      No_S_arcs++;
	      S_Arcs[No_S_arcs].from = Root ;
	      S_Arcs[No_S_arcs].to = np->phase;
	      S_Arcs[No_S_arcs].cost = -1*np->bounds[LOWER];
	      No_S_arcs++;
	    } /* if reduce */

	}

      if(np->nao ==1) /* Mirror node  */
	{
	  for(w1 = np->fout;w1 != NULL;w1 = w1->next)
	    {
	      n2 = w1->el;
	      np2 = nodes[n2];  /* only arcs from non bound nodes in Reduce mode*/
	      if((Red_Mode == SHENOY)||(np2->IsFixed == 0))
		{
		  S_Arcs[No_S_arcs].from = np2->phase ;
		  S_Arcs[No_S_arcs].to = np->Id_m;
		  S_Arcs[No_S_arcs].cost = np->Maxm - w1->wt;
		  No_S_arcs++;
		}
	    }
	  if((Red_Mode == REDUCE)&&(n != HOST))
	    {
	      S_Arcs[No_S_arcs].from = np->Id_m;
	      S_Arcs[No_S_arcs].to = Root;
	      S_Arcs[No_S_arcs].cost = np->Um;
	      No_S_arcs++;
	      S_Arcs[No_S_arcs].from = Root ;
	      S_Arcs[No_S_arcs].to = np->Id_m;
	      S_Arcs[No_S_arcs].cost = -1*np->Lm;
	      No_S_arcs++;
	    }

	}

    } /* all gates */

}

/********************************************
transfers the r(v) values form simplex network to C-graph
***********************************************/

int read_simplex( int Red_Mode)
{
  int n,i;
  NODETYPE *np;


  /* reset phase of PIO */
  for(i=0;i<nopin;i++)
    {
      nodes[primary_input[i]]->phase = 0;
    }
  for(i=0;i<nopout;i++)
    {
      nodes[primary_output[i]]->phase = 0;
    }

  if(Red_Mode == REDUCE)for(i=0;i<nogates;i++)
    {
      n = gates[i];
      np = nodes[n];
      if(np->level == 0){
	np->phase = np->bounds[LOWER];
      }
      if((np->Ism == 1)&&(np->nao == 0)){
	np->Xm = np->Um;
      }
    }

  for(i = 0;i<No_S_nodes;i++)
    {
      n = S_nodes[i].ckt_id;
      if(n >=0)
	{
	  nodes[n]->phase = S_nodes[i].dual;
#ifdef SAFE
	  if((nodes[n]->level == 0)&& n != HOST){
	    printf(" ERR constant gate %d[%s] of Type %s had value read as %d from node %d \n",n,nodes[n]->name,
		   Node_Type[nodes[n]->type],S_nodes[i].dual,i);
	  }
#endif
	}
      else{ 
	n = -1*n; 
	n--;  
	nodes[n]->Xm = S_nodes[i].dual;
      }
    }

  /* chaeck vars in bound */
  if(Red_Mode == REDUCE)for(i=0;i<nogates -1;i++)
    {
      n = gates[i];
      np = nodes[n];
      if((np->phase < np->bounds[LOWER])|| (np->phase > np->bounds[UPPER]))
	{ 
	  printf("ERR {read_simplex} Retime of gate [%d]%s T %s is %d U %d L %d level %d\n",n,np->name,Node_Type[np->type],np->phase,np->bounds[UPPER],np->bounds[LOWER],np->level);
	  exit(0);
	}

      if((np->Ism == 1)&&((np->Xm < np->Lm)|| (np->Xm > np->Um)))
	{ 
	  printf("ERR {read_simplex} Retime of Mirror gate [%d]%s is %d U %d L %d nao %d\n",n,np->name,np->Xm,
		 np->Um,np->Lm,np->nao);
	  exit(0);
	}

    }

  if(0)for(i=0;i<nogates;i++)fprintf(fperr," %s %d \n",nodes[gates[i]]->name,nodes[gates[i]]->phase);
  free(S_nodes);
  return(Min_cost);

}



/********************************************
Just performs some simple checks on the simplex network
***********************************************/

void check_nodes_S_Arcs()
{
  int i,j;
  Q *q1;


  printf("checking NODES  \n");
  for(i = 0;i<No_S_nodes;i++)
    {
      j = S_nodes[i].orient;
      if(!((j == -1)||(j == 1))){
	printf(" ERR in node %d orient %d \n",i,j);
	exit(0);
      }
      j = S_nodes[i].flow;
      if(j <0){
	printf(" ERR in node %d flow %d \n",i,j);
	exit(0);
      }
      j = S_nodes[i].level;
      if((j <0)){
	printf(" ERR in node %d level %d \n",i,j);
	exit(0);
      }
      j = S_nodes[i].parent;
      if((j <0)|| (j >= No_S_nodes)){
	printf(" ERR in node %d parent %d \n",i,j);
	exit(0);
      }
      for(q1 = S_nodes[i].child;q1 != NULL;q1 = q1->next)
	{
	  j =q1->el;
	  if((j <0)|| (j >= No_S_nodes)){
	    printf(" ERR in node %d child %d \n",i,j);
	    exit(0);
	  }
	}
    }
  printf("checked %d nodes\n",No_S_nodes);

  printf("checking S_Arcs  \n");
  fflush(stdout);
  for(i = 0;i<No_S_arcs;i++)
    {
      j = S_Arcs[i].to;
      if((j <0)|| (j >= No_S_nodes)){
	printf(" ERR in arc %d to %d \n",i,j);
	exit(0);
      }
      j = S_Arcs[i].from;
      if((j <0)|| (j >= No_S_nodes)){
	printf(" ERR in arc %d from %d \n",i,j);
	exit(0);
      }

    }
  printf("checked %d S_Arcs\n",No_S_arcs);
  fflush(stdout);
}

/********************************************
make_simplex_tree

make the spanning tree as required my the network simplex algorithm

WILL USE phase to store id of S_node
         level == 1 for variable, 0 for fixed node
	 nao == 1 if mirror is variable, 0 otherwise
	 Id_m to store id in S_node (for mirror vertex)
	 nai to store supply;
	 
***********************************************/
void make_simplex_tree(int Red_Mode)
{

  int i,n,diff,n2;
  NODETYPE *np, *np2;
  int Init_cost;
  int exp_arcs;/* expected no of arcs */
  int exp_nodes;/* expected no of vars */
  int constant_obj =0;  /* constant part of obj for red_mode = 2 */
  int host_obj=0;  /* - of sum of coeff of all vars in obj to be used as obj of host */
  int cost;
  P_SIZE prob_size;
  int exp_m_arcs;  /* no of arc expected to mirror for a node */
  Q *q2;


  /* MAKE TREE NOW */
  S_nodes[Root].parent = Root;
  S_nodes[Root].level = 0;
  S_nodes[Root].dual = 0;
  S_nodes[Root].flow = 0;
  S_nodes[Root].orient = 1;
  Min_cost = 0;
  Init_cost = 1000;
  for(i=0;i<(nogates -1);i++)
    {
      np = nodes[gates[i]];
      if(np->level == 1)/*   make tree out of bound arcs for gates */
	{
	  n = nodes[gates[i]]->phase;
	  S_nodes[n].parent = Root;
	  S_nodes[n].level = 1;
	  diff = S_nodes[n].supply;

	  if(diff >= 0)
	    {
	      if(Red_Mode == REDUCE) { 
		Init_cost = np->bounds[UPPER]; 
	      }
	      else Init_cost = 1000;
	      S_nodes[n].orient = 1;
	      S_nodes[n].flow = diff;
	      Min_cost += diff*Init_cost;
	      S_nodes[n].dual = Init_cost;
	      /* add other bound arc */
	    }
	  if(diff < 0)
	    {
	      if(Red_Mode == REDUCE) { 
		Init_cost = -1*np->bounds[LOWER];  
	      }
	      else Init_cost = 1000;
	      diff = -1*diff; /* negative flow indiacated by orient */
	      S_nodes[n].orient = -1;
	      S_nodes[n].flow = diff;
	      Min_cost += diff*Init_cost;
	      S_nodes[n].dual = 0 - Init_cost;
	    }
	  q2 = (Q *)MALLOC(1,sizeof(Q));
	  q2->el = n;
	  q2->next = S_nodes[Root].child ;
	  S_nodes[Root].child = q2;
	  S_nodes[Root].nochild++;
	}
      if(np->nao == 1)/*   make tree out of bound arcs for gates */
	{
	  n = nodes[gates[i]]->Id_m;
	  S_nodes[n].parent = Root;
	  S_nodes[n].level = 1;
	  if(Red_Mode == REDUCE)Init_cost = -1*np->Lm;
	  S_nodes[n].orient = -1;
	  S_nodes[n].flow = 1;
	  Min_cost += 1*Init_cost;
	  S_nodes[n].dual = 0 - Init_cost;

	  q2 = (Q *)MALLOC(1,sizeof(Q));
	  q2->el = n;
	  q2->next = S_nodes[Root].child ;
	  S_nodes[Root].child = q2;
	  S_nodes[Root].nochild++;
	}
    }

#ifdef DEBUG
  printf(" TREE made is min cost %d Root %d  \n",Min_cost,Root);
  if(0)print_S_Arcs();
  if(0)print_S_nodes();
#endif
  if(Red_Mode == SHENOY)prob_size.no_con = No_S_arcs + No_S_nodes -1;
  if(Red_Mode == REDUCE)prob_size.no_con = No_S_arcs ;
  prob_size.no_var = No_S_nodes;
  return;

}


/********************************************
get_max_cost_in_simplex
***********************************************/
int get_max_cost_in_simplex()
{
  int i,j,max=0;

  printf("S_Arcs  are \n");
  for(i = 0;i<No_S_arcs;i++)
    {
      if(max < S_Arcs[i].cost)max = S_Arcs[i].cost;
    }
  printf(" Max cost in S_Arcs is %d \n",max);
  return(max);
}


/********************************************
print_S_nodes

just for debugging
***********************************************/
void print_S_nodes()
{
  int i,j,tt,m;
  Q *q1;

  printf(" NODES are \n");
  for(i = 0;i<No_S_nodes;i++)
    {
      tt = S_nodes[i].ckt_id;
      if(tt <0){ 
	tt = -1*tt;
	tt--;
	m = 1;
      } else m = 0;
      printf(" Node %d (ckt id %d ie %s M%d ) Sup %d P %d L %d dual %d F %d Or %d \n \t %d child are  ",i,S_nodes[i].ckt_id,
	     nodes[tt]->name,m,S_nodes[i].supply,S_nodes[i].parent,S_nodes[i].level,S_nodes[i].dual,S_nodes[i].flow,S_nodes[i].orient,
	     S_nodes[i].nochild);
      for(q1 = S_nodes[i].child;q1 != NULL;q1 = q1->next)printf(" %d  ",q1->el);
      printf("\n");

    }
  printf("\n");
  fflush(stdout);
}


/********************************************
print_S_Arcs
just for debugging
***********************************************/
void print_S_Arcs()
{
  int i,j;

  printf("S_Arcs  are \n");
  for(i = 0;i<No_S_arcs;i++)
    {
      printf(" arc %d from %d  to %d cost %d  \n",i,S_Arcs[i].from,S_Arcs[i].to,S_Arcs[i].cost);
    }
  printf("\n");
  fflush(stdout);
}

/*********EOF**********/
